unit OneInstance;

{>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>}
{> Komponente: TOneInstance                                                                >}
{> Beschr.   : Verhindert mehrere Instanzen der Anwendung                                  >}
{> Properties: Enabled: berwachung aktiv                                                  >}
{>             AutoBringToFront: Nach Wechsel erste Instanz in den Vordergrund bringen     >}
{>             OnExists: Wird in der zweiten Instanz ausgelst, wenn die erste luft       >}
{>             OnSwitch: Wird in der ersten Instanz ausgelst, nachdem die zweite die      >}
{>                       Kommandozeilen-Parameter bermittelt hat und beendet wurde        >}
{> Version   : 1.0, 18.08.2004                                                             >}
{> Autor     : Daniel Pauli                                                                >}
{> Kontakt   : daniel.pauli@gmx.ch                                                         >}
{> Lizenz    : Frei verfgbar fr Delphi-PRAXiS-Mitglieder                                 >}
{>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>}

interface

uses
  Windows, Messages, SysUtils, Classes, Controls, Forms;

type
  TOnExists = procedure(var Switch: Boolean) of object;
  TOnSwitch = procedure(Parameters: TStrings) of object;

  TOneInstance = class(TComponent)
  private
    FOwner: TWinControl;
    FEnabled: Boolean;
    FAutoBringToFront: Boolean;
    FOnExists: TOnExists;
    FOnSwitch: TOnSwitch;
    FFirstInst: THandle;
    FSecondInst: THandle;
    FOldWndProc: TWndMethod;
    procedure NewWndProc(var Message: TMessage);
    procedure SendParameters(const Handle: THandle);
    procedure DoSwitch(const ParamStr: string);
  protected
    procedure Loaded; override;
  public
    constructor Create(AOwner: TComponent); override;
  published
    property Enabled: Boolean read FEnabled write FEnabled default True;
    property AutoBringToFront: Boolean read FAutoBringToFront write FAutoBringToFront default True;
    property OnExists: TOnExists read FOnExists write FOnExists;
    property OnSwitch: TOnSwitch read FOnSwitch write FOnSwitch;
  end;

procedure Register;

function IsFirstInstance: Boolean;
function ForceForegroundWindow(hwnd: THandle): Boolean;

implementation

{$R *.dcr}

var
  WM_COMMUNICATE: Cardinal;
  GUID: string;

const
  GET_HANDLE     = $FAF0;
  CHECK_SWITCH   = $FAF1;
  GET_PARAMS     = $FAF2;
  BRING_TO_FRONT = $FAF4;

procedure Register;
begin
  RegisterComponents('Eigene Komponenten', [TOneInstance]);
end;

constructor TOneInstance.Create(AOwner: TComponent);
begin
  inherited;
  FAutoBringToFront:= True;
  FEnabled:= True;
  if not (csDesigning in ComponentState) then
  begin
    WM_COMMUNICATE:= RegisterWindowMessage(PChar(GUID));
    FOwner:= AOwner as TWinControl;
    FOldWndProc:= FOwner.WindowProc;
    FOwner.WindowProc:= NewWndProc;
  end;
end;

function IsFirstInstance: Boolean;
begin
  CreateMutex(nil, True, PChar(GUID));
  Result:= GetLastError <> ERROR_ALREADY_EXISTS;
end;

procedure TOneInstance.Loaded;
begin
  inherited;
  if not (csDesigning in ComponentState) then
  begin
    if not IsFirstInstance then
      { 2. Instanz: Handle der 1. Instanz anfordern }
      SendMessage(HWND_BROADCAST, WM_COMMUNICATE, FOwner.Handle, GET_HANDLE);
  end;
end;

procedure TOneInstance.NewWndProc(var Message: TMessage);
var
  Switch: Boolean;
  S: string;
  Command: Integer;
begin
  with Message do
    if Msg = WM_COMMUNICATE then
    begin
      if THandle(wParam) <> FOwner.Handle then
      begin
        if lParam = GET_HANDLE then
        begin
          { *** 1. Instanz *** }
          { Falls Komponente aktiv, eigenes Handle zurcksenden und Switch einleiten }
          ReplyMessage(1);
          FSecondInst:= THandle(wParam);
          if FEnabled then
          begin
            Command:= CHECK_SWITCH;
            if Assigned(FOnSwitch) then Command:= Command or GET_PARAMS;
            if FAutoBringToFront then Command:= Command or BRING_TO_FRONT;
            SendMessage(FSecondInst, WM_COMMUNICATE, FOwner.Handle, Command);
          end;
        end else
        if (lParam and CHECK_SWITCH) = CHECK_SWITCH then
        begin
          { *** 2. Instanz *** }
          { Erhlt Handle der ersten Instanz und prft, ob sie selbst beendet werden soll }
          ReplyMessage(1);
          FFirstInst:= THandle(wParam);
          Switch:= True;
          if Assigned(FOnExists) then FOnExists(Switch);
          if Switch then
          begin
            Application.ShowMainForm:= False;
            { Falls angefordert, Parameter an erste Instanz senden }
            if (lParam and GET_PARAMS) = GET_PARAMS then SendParameters(FFirstInst);
            if (lParam and BRING_TO_FRONT) = BRING_TO_FRONT then ForceForegroundWindow(FFirstInst);
            { Bye }
            Application.Terminate;
          end;
        end;
      end;
      Exit;   // Selbstdefinierte Message braucht nicht weitergereicht zu werden
    end else
    if Msg = WM_COPYDATA then
    begin
      if THandle(wParam) = FSecondInst then
      begin
        { *** 1. Instanz ***}
        { OnSwitch mit den empfangenen Parametern auslsen }
        ReplyMessage(1);
        with PCopyDataStruct(lParam)^ do
          SetString(S, PChar(lpData), cbData);
        DoSwitch(S);
        Exit;
      end;
    end;
  FOldWndProc(Message);
end;

procedure TOneInstance.SendParameters(const Handle: THandle);
var
  MyCopyDataStruct: TCopyDataStruct;
  S: string;
  I: Integer;
begin
  S:= '';
  for I:= 1 to ParamCount do
    S:= S + ParamStr(I) + '|';
  with MyCopyDataStruct do
  begin
    dwData:= 0;
    lpData:= PChar(S);
    cbData:= StrLen(lpData);
  end;
  SendMessage(Handle, WM_COPYDATA, FOwner.Handle, Integer(@MyCopyDataStruct));
end;

procedure TOneInstance.DoSwitch(const ParamStr: string);
var
  Parameters: TStrings;
  I, J: Integer;
begin
  Parameters:= TStringList.Create;
  try
    J:= 1;
    while J < Length(ParamStr) do
    begin
      I:= J;
      repeat Inc(J) until ParamStr[J] = '|';
      Parameters.Add(Copy(ParamStr, I, J - I));
      Inc(J);
    end;
    if Assigned(FOnSwitch) then FOnSwitch(Parameters);
  finally
    FreeAndNil(Parameters);
  end;
end;

function ForceForegroundWindow(hwnd: THandle): Boolean;
{ Quelle: SwissDelphiCenter.ch }
const 
  SPI_GETFOREGROUNDLOCKTIMEOUT = $2000; 
  SPI_SETFOREGROUNDLOCKTIMEOUT = $2001; 
var 
  ForegroundThreadID: DWORD; 
  ThisThreadID: DWORD; 
  timeout: DWORD; 
begin 
  if IsIconic(hwnd) then ShowWindow(hwnd, SW_RESTORE); 

  if GetForegroundWindow = hwnd then Result := True 
  else 
  begin 
    // Windows 98/2000 doesn't want to foreground a window when some other 
    // window has keyboard focus 

    if ((Win32Platform = VER_PLATFORM_WIN32_NT) and (Win32MajorVersion > 4)) or 
      ((Win32Platform = VER_PLATFORM_WIN32_WINDOWS) and 
      ((Win32MajorVersion > 4) or ((Win32MajorVersion = 4) and 
      (Win32MinorVersion > 0)))) then 
    begin 
      // Code from Karl E. Peterson, www.mvps.org/vb/sample.htm 
      // Converted to Delphi by Ray Lischner 
      // Published in The Delphi Magazine 55, page 16 

      Result := False; 
      ForegroundThreadID := GetWindowThreadProcessID(GetForegroundWindow, nil); 
      ThisThreadID := GetWindowThreadPRocessId(hwnd, nil); 
      if AttachThreadInput(ThisThreadID, ForegroundThreadID, True) then 
      begin 
        BringWindowToTop(hwnd); // IE 5.5 related hack 
        SetForegroundWindow(hwnd); 
        AttachThreadInput(ThisThreadID, ForegroundThreadID, False); 
        Result := (GetForegroundWindow = hwnd); 
      end; 
      if not Result then 
      begin 
        // Code by Daniel P. Stasinski 
        SystemParametersInfo(SPI_GETFOREGROUNDLOCKTIMEOUT, 0, @timeout, 0); 
        SystemParametersInfo(SPI_SETFOREGROUNDLOCKTIMEOUT, 0, TObject(0), 
          SPIF_SENDCHANGE); 
        BringWindowToTop(hwnd); // IE 5.5 related hack 
        SetForegroundWindow(hWnd); 
        SystemParametersInfo(SPI_SETFOREGROUNDLOCKTIMEOUT, 0, TObject(timeout), SPIF_SENDCHANGE); 
      end; 
    end 
    else 
    begin 
      BringWindowToTop(hwnd); // IE 5.5 related hack 
      SetForegroundWindow(hwnd); 
    end; 

    Result := (GetForegroundWindow = hwnd); 
  end; 
end; { ForceForegroundWindow }

procedure MakeGUID;
var
  S: string;
  I: Integer;
begin
  S:= LowerCase(Application.ExeName);
  for I:= 1 to Length(S) do
    if S[I] = '\' then S[I]:= '/';
  GUID:= '#OneInst:' + S;
end;

initialization
  MakeGUID;

end.
