unit InstCtrl;

interface

{$IF CompilerVersion >= 14.0} {$DEFINE DELPHI_6_UP} {$IFEND}
{$IF CompilerVersion = 20.0} {$DEFINE DELPHI_2009} {$IFEND}
{$IF CompilerVersion >= 20.0} {$DEFINE DELPHI_2009_UP} {$IFEND}

uses
  SysUtils, Classes, Messages, Windows, Forms;

type
  PFirstInstanceMap = ^TFirstInstanceMap;
  TFirstInstanceMap = packed record
    FirstWinHdl: HWND;
    MaxNewInstances: ShortInt;
    NewInstancesCount: Byte;
  end;

  PNewInstanceMap = ^TNewInstanceMap;
  TNewInstanceMap = packed record
    CommandLineWSize: Cardinal;
  end;

  TPOnNewInstanceLoad = procedure (NewAppHandle: Cardinal; NewCommandLineW: PWideChar)
    of object;

  TOneInstance = class(TComponent)
  private
    FMaxNewInstances: ShortInt;
    FMsgTransportTimeout: Cardinal;
    function GetAbout: WideString;
    procedure SetAbout(const Value: WideString);
    function GetMaxNewInstances: ShortInt;
    procedure SetMaxNewInstances(const Value: ShortInt);
    { Private declarations }
  protected
    FOnNewInstanceLoad: TPOnNewInstanceLoad;

    FFirstMap: PFirstInstanceMap;
    FFirstMapName: WideString;
    FFirstMapHandle: Cardinal;

    FWindowHandle: Cardinal;
    procedure WindowProc(var Message: TMessage); virtual;
    { Protected declarations }
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    { Public declarations }
  published
    property About: WideString read GetAbout write SetAbout;
    property MaxNewInstances: ShortInt read GetMaxNewInstances
      write SetMaxNewInstances;
      { This property means the number of new instances which
      can send message (contains new parameters) to first instance,
      does not mean the number of new instances can be executed.
        Set this property < 0 to be unlimited.}
    property MsgTransportTimeout: Cardinal read FMsgTransportTimeout
      write FMsgTransportTimeout default 3000;
    property OnNewInstanceLoad: TPOnNewInstanceLoad read FOnNewInstanceLoad
      write FOnNewInstanceLoad;
    { Published declarations }
  end;

procedure Register;

function WParamCount(const CommandLineW: PWideChar): Integer;
function WParamStr(const CommandLineW: PWideChar; Index: Integer): WideString;

{$IFNDEF UNICODE}
function WStrAlloc(Size: Cardinal): PWideChar;
procedure WStrDispose(S: PWideChar);
{$ENDIF}

function WStringToPointer(WS: WideString; var P: Pointer): Integer;
function PointerToWString(P: Pointer): WideString;

implementation

uses
  ShellAPI;

const
  FirstMapName = 'TOneInstance';
  ShareStrMapName = 'TOneInstance::ShareStrMapName';
  Msg_NewParamStr = WM_USER + 101;

var
  SelfInstance: Byte = 0;

  // Common variables
  MapName: WideString;
  NewMap: PNewInstanceMap;
  NewMapHandle: Cardinal;

procedure Register;
begin
  RegisterComponents('Externe', [TOneInstance]);
end;
//===

function WParamCount(const CommandLineW: PWideChar): Integer;
begin
  LocalFree(Cardinal(ShellAPI.CommandLineToArgvW(CommandLineW, Result)));
end;

function WParamStr(const CommandLineW: PWideChar; Index: Integer): WideString;
var
  ArgvW: PPWideChar;
  PrCount: Integer;
begin
  Result := '';

  ArgvW := CommandLineToArgvW(CommandLineW, PrCount);
  if (Index >= 0) and (Index < PrCount) then
  begin
    Inc(ArgvW, Index);
    Result := ArgvW^;
    Dec(ArgvW, Index);
  end;

  LocalFree(Cardinal(ArgvW));
end;

{$IFNDEF UNICODE}
function WStrAlloc(Size: Cardinal): PWideChar;
begin
  Size := Size * SizeOf(WideChar) + SizeOf(Cardinal);
  GetMem(Result, Size);
  Cardinal(Pointer(Result)^) := Size;
  Result := Ptr(Cardinal(Result) + SizeOf(Cardinal));
end;

procedure WStrDispose(S: PWideChar);
begin
  if S <> nil then
  begin
    S := Ptr(Cardinal(S) - SizeOf(Cardinal));
    FreeMem(S, Cardinal(Pointer(S)^));
  end;
end;
{$ENDIF}

function WStringToPointer(WS: WideString; var P: Pointer): Integer;
var
  ps: PAnsiChar;
  str: AnsiString;
begin
  str := utf8Encode(WS);
  {$IFDEF UNICODE}
  ps := AnsiStrAlloc(Length(str) + 1);
  {$ELSE}
  ps := StrAlloc(Length(str) + 1);
  {$ENDIF}
  SysUtils.StrCopy(ps, PAnsiChar(str)); // use SysUtils for AnsiString

  Result := StrLen(ps) + 1;// * SizeOf(Char);
  GetMem(p, Result + SizeOf(Cardinal));
  Cardinal(p^) := Result;
  p := Ptr(Cardinal(p) + SizeOf(Cardinal));
  CopyMemory(p, ps, Result);
  StrDispose(ps);
  p := Ptr(Cardinal(p) - SizeOf(Cardinal));
  Inc(Result, SizeOf(Cardinal));
end;

function PointerToWString(P: Pointer): WideString;
var
  sz: Cardinal;
  ps: PAnsiChar;
begin
  sz := Cardinal(p^);
  P := Ptr(Cardinal(P) + SizeOf(Cardinal));
  {$IFDEF UNICODE}
  ps := AnsiStrAlloc(sz);
  {$ELSE}
  ps := StrAlloc(sz);
  {$ENDIF}
  CopyMemory(ps, P, sz);
  {$IFDEF UNICODE}
  Result  := utf8ToWideString(ps);
  {$ELSE}
  Result := utf8Decode(ps);
  {$ENDIF}
  StrDispose(ps);
  P := Ptr(Cardinal(P) - SizeOf(Cardinal));
end;

{ TOneInstance }

constructor TOneInstance.Create(AOwner: TComponent);
var
  StrBuf, p: Pointer;
  AppHandle: Cardinal;
  StrMapHandle: Cardinal;
  FindData: _WIN32_FIND_DATAW;
  MsgResult: Cardinal;
begin
  Inc(SelfInstance);

  if not(AOwner is TForm) then
    raise Exception.Create('AOwner must be TForm or descendant');

  if SelfInstance > 1 then
  begin
    raise Exception.Create(
      Format('Only one copy of %s can be used on one application', [Self.ClassName]));
  end;

  inherited;

  FMsgTransportTimeout := 3000;

  FindClose(FindFirstFileW(PWideChar(WParamStr(GetCommandLineW(), 0)), FindData));
  FFirstMapName := WideFormat('%s::%d', [FirstMapName, FindData.nFileSizeLow]);

  if not (csDesigning in ComponentState) then
  begin
    FFirstMapHandle := OpenFileMappingW(
      FILE_MAP_ALL_ACCESS,
      False,
      PWideChar(FFirstMapName));
    if FFirstMapHandle <> 0 then
      FFirstMap := MapViewOfFile(FFirstMapHandle, FILE_MAP_ALL_ACCESS, 0, 0, 0);

    if FFirstMap = nil then
    begin
      FFirstMapHandle := CreateFileMappingW(
        INVALID_HANDLE_VALUE, // or MAXDWORD, $FFFFFFFF: use paging file
        nil,               // no security attributes
        PAGE_READWRITE,    // read/write access
		    0,                 // size: high 32-bits
        SizeOf(FFirstMap^), // size: low 32-bits
		    PWideChar(FFirstMapName));
      FFirstMap := MapViewOfFile(FFirstMapHandle, FILE_MAP_ALL_ACCESS, 0, 0,
        SizeOf(FFirstMap^));
      ZeroMemory(FFirstMap, SizeOf(FFirstMap^));

      {$IFDEF DELPHI_6_UP}
      FWindowHandle := Classes.AllocateHWnd(WindowProc);
      {$ELSE}
      FWindowHandle := AllocateHWnd(WindowProc);
      {$ENDIF}

      with FFirstMap^ do
      begin
        FirstWinHdl := FWindowHandle;
        MaxNewInstances := FMaxNewInstances;
        NewInstancesCount := 0;
      end;
    end else
    begin
      with FFirstMap^ do
      begin
        FMaxNewInstances := MaxNewinstances;
        Inc(NewInstancesCount);
        if MaxNewInstances >= 0 then
          // Set MaxNewInstances < 0 to be unlimited.
          if NewInstancesCount > MaxNewInstances then
          begin
            Dec(NewInstancesCount);
            ExitProcess(0);
          end;
      end;

      with Application do
      begin
        ShowMainForm := False;
        ShowWindow(Handle, SW_HIDE);
        AppHandle := Handle;
      end;

      MapName := WideFormat('%s::%d', [ShareStrMapName, AppHandle]);

      NewMapHandle := CreateFileMappingW(INVALID_HANDLE_VALUE, nil,
        PAGE_READWRITE, 0, SizeOf(NewMap^), PWideChar(MapName));
      NewMap := MapViewOfFile(NewMapHandle, FILE_MAP_ALL_ACCESS, 0, 0,
        SizeOf(NewMap^));
      NewMap^.CommandLineWSize :=
        WStringToPointer(WideString(GetCommandLineW()), p);
        // use WStringToPointer() to save memory

      MapName := WideFormat('%s::%d::%d', [ShareStrMapName, AppHandle, AppHandle]);

      with NewMap^ do
      begin
        StrMapHandle := CreateFileMappingW(INVALID_HANDLE_VALUE, nil,
          PAGE_READWRITE, 0, CommandLineWSize, PWideChar(MapName));
        StrBuf := MapViewOfFile(StrMapHandle, FILE_MAP_ALL_ACCESS,
          0, 0, CommandLineWSize);
        CopyMemory(StrBuf, p, CommandLineWSize);
        FreeMem(p, CommandLineWSize);
      end;

      UnMapViewOfFile(NewMap);
      UnMapViewOfFile(StrBuf);

      SendMessageTimeout(FFirstMap^.FirstWinHdl, Msg_NewParamStr,
        AppHandle, Msg_NewParamStr,
          //Set LParam = Msg_NewParamStr for testing
        SMTO_BLOCK or SMTO_ABORTIFHUNG, FMsgTransportTimeout, MsgResult);
      { Use SendMessage() and wait for the first instance
      processes the file mapping, then we close it.
        Except on the first instance is hung, we just have to wait in
      FMsgTransportTimeout milliseconds.}

      CloseHandle(NewMapHandle);
      CloseHandle(StrMapHandle);

      Dec(FFirstMap^.NewInstancesCount);
      ExitProcess(0);
    end;
  end;
end;

destructor TOneInstance.Destroy;
begin
  if not (csDesigning in ComponentState) then
  begin
    if FFirstMap <> nil then UnmapViewOfFile(FFirstMap);
    if FFirstMapHandle <> 0 then CloseHandle(FFirstMapHandle);

    if IsWindow(FWindowHandle) then
      {$IFDEF DELPHI_6_UP}
      Classes.DeallocateHWnd(FWindowHandle);
      {$ELSE}
      DeallocateHWnd(FWindowHandle);
      {$ENDIF}
  end;

  Dec(SelfInstance);

  inherited;
end;

function TOneInstance.GetAbout: WideString;
const
  BkLine = #13#10;
begin
{ History:

  1.50: July 05, 2009;
        Add the property MsgTransportTimeout;
  1.40: July 02, 2009;
}
  Result := WideFormat('%s%s%s'
      + 'Version: 1.50%s'
      + 'Build date: July 05, 2009%s'
      + 'License: Free for any usage%s'
      + 'IDE: Delphi 5/6/7/2009%s%s'
      + 'Opera blog: guy8th%s'
      + 'freeman_0408@yahoo.com',
    [ClassName, BkLine, BkLine,
      BkLine,
      BkLine,
      BkLine,
      BkLine, BkLine,
      BkLine]);
end;

function TOneInstance.GetMaxNewInstances: ShortInt;
begin
  if csDesigning in ComponentState then
    Result := FMaxNewInstances
  else
    Result := FFirstMap^.MaxNewInstances;
end;

procedure TOneInstance.SetAbout(const Value: WideString);
begin
  //
end;

procedure TOneInstance.SetMaxNewInstances(const Value: ShortInt);
begin
  if csDesigning in ComponentState then
    FMaxNewInstances := Value
  else
    FFirstMap^.MaxNewInstances := Value;
end;

procedure TOneInstance.WindowProc(var Message: TMessage);
var
  StrBuf: Pointer;
  StrMapHandle: Cardinal;
begin
  with Message do
    if Msg = Msg_NewParamStr then
    begin
      if Message.LParam <> Msg_NewParamStr then Exit;
        { Test the message Msg_NewParamStr,
        was it send by itself or another application? }
        
      try
        if (FFirstMap <> nil) then
        begin
          MapName := WideFormat('%s::%d', [ShareStrMapName, WParam]);
          NewMaphandle := OpenFileMappingW(FILE_MAP_ALL_ACCESS, False,
            PWideChar(MapName));
          MapName := WideFormat('%s::%d::%d', [ShareStrMapName, WParam, WParam]);
          StrMapHandle := OpenFileMappingW(FILE_MAP_ALL_ACCESS, False,
            PWideChar(MapName));
          if InSendMessage() then ReplyMessage(Integer(True));
            { After open the file mapping,
            we should reply this message for the newest istance to exits
            without waiting for our processing here. }

          if (NewMapHandle <> 0) and (StrMapHandle <> 0)
          then
          begin
            NewMap := MapViewOfFile(NewMapHandle, FILE_MAP_ALL_ACCESS, 0, 0,
              SizeOf(NewMap^));
            StrBuf := MapViewOfFile(StrMapHandle, FILE_MAP_ALL_ACCESS, 0, 0,
              NewMap^.CommandLineWSize);

            UnMapViewOfFile(NewMap);
            CloseHandle(NewMaphandle);

            if StrBuf <> nil then
            begin
              if (Assigned(FOnNewInstanceLoad)) then
                FOnNewInstanceLoad(WParam, PWideChar(PointerToWString(StrBuf)));
              UnMapViewOfFile(StrBuf);
            end;

            CloseHandle(StrMapHandle);
          end;
        end;
      except
        Application.HandleException(Self);
      end
    end else
    begin
      Result := DefWindowProc(FWindowHandle, Msg, wParam, lParam);
    end;
end;

initialization

finalization

end.

