/*
    Noise Reduction Filter for VirtualDub
	by tHE fISH (fish@everymail.net)

    based on the Antiflicker Filter for VirtualDub which is
	Copyright (C) 1999-2000 Donald A. Graft

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

	The author can be contacted at:
	Donald Graft
	neuron2@home.com.
*/

#include <windows.h>
#include <commctrl.h>
#include <stdio.h>
#include <math.h>
#include <crtdbg.h>

#include "ScriptInterpreter.h"
#include "ScriptError.h"
#include "ScriptValue.h"

#include "resource.h"
#include "filter.h"

///////////////////////////////////////////////////////////////////////////

long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int RunProc(const FilterActivation *fa, const FilterFunctions *ff);
int StartProc(FilterActivation *fa, const FilterFunctions *ff);
int EndProc(FilterActivation *fa, const FilterFunctions *ff);
long ParamProc(FilterActivation *fa, const FilterFunctions *ff);
int MyInitProc(FilterActivation *fa, const FilterFunctions *ff);
int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);
bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen);

///////////////////////////////////////////////////////////////////////////

typedef struct MyFilterData {
	int					softening;
	unsigned char		*old_data;
} MyFilterData;

bool FssProc(FilterActivation *fa, const FilterFunctions *ff, char *buf, int buflen) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	_snprintf(buf, buflen, "Config(%d)", mfd->softening);

	return true;
}

void ScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc) {
	FilterActivation *fa = (FilterActivation *)lpVoid;
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->softening	= argv[0].asInt();
}

ScriptFunctionDef func_defs[]={
	{ (ScriptFunctionPtr)ScriptConfig, "Config", "0i" },
	{ NULL },
};

CScriptObject script_obj={
	NULL, func_defs
};

struct FilterDefinition filterDef_tutorial = {

	NULL, NULL, NULL,		// next, prev, module
	"denoise (1.0)",   	    // name
	"Removes noise from images\n\nBased on the Deflicker plugin by Donald A. Graft",
							// desc
	"tHE fISH", 		    // maker
	NULL,					// private_data
	sizeof(MyFilterData),	// inst_data_size

	MyInitProc,				// initProc
	NULL,					// deinitProc
	RunProc,				// runProc
	NULL,				    // paramProc
	ConfigProc, 			// configProc
	StringProc, 			// stringProc
	StartProc,				// startProc
	EndProc,				// endProc

	&script_obj,			// script_obj
	FssProc,				// fssProc

};

#define MAX_SOFTENING 31

///////////////////////////////////////////////////////////////////////////

int StartProc(FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	mfd->old_data = new unsigned char [fa->src.Size()];
	memset (mfd->old_data, 0, fa->src.Size());

	return 0;
}

int EndProc(FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	delete[] mfd->old_data;
	mfd->old_data = NULL;

	return 0;
}

int RunProc(const FilterActivation *fa, const FilterFunctions *ff)
{
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;
	const PixDim	w = fa->src.w;
	const PixDim	h = fa->src.h;
	Pixel32 *src, *dst;
	int y;

	src = (Pixel32 *)((char *)fa->src.data);
	dst = (Pixel32 *)((char *)fa->dst.data);
	for (y = 0; y < h; y++)
	{
		memcpy(dst, src, fa->src.w * sizeof(Pixel32));
		src	= (Pixel32 *)((char *)src + fa->src.pitch);
		dst	= (Pixel32 *)((char *)dst + fa->dst.pitch);
	}


	/* Temporal softening phase. Adapted from code by Steven Don. */
	unsigned char *src1, *src2;
	long diff, ofs, sum;

	if (mfd->softening == 0) return 0;

	ofs = fa->src.Size();
	src1 = (unsigned char *) fa->dst.data;
	src2 = (unsigned char *) mfd->old_data;
	do
	{
		diff = abs(*src1 - *src2);
		if (diff < mfd->softening)
		{
			if (diff > (mfd->softening >> 1))
			{
				sum = *src1 + *src1 + *src2;
				*src2 = sum / 3;
			}
		}
		else
		{
			*src2 = *src1;
		}
		*src1 = *src2;
		src1++; src2++;
	} while (--ofs);

	return 0;
}

extern "C" __declspec(dllexport) int VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat);
extern "C" __declspec(dllexport) void VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff);

static FilterDefinition *fd_tutorial;

int VirtualdubFilterModuleInit2(FilterModule *fm, const FilterFunctions *ff, int& vdfd_ver, int& vdfd_compat) {
	if ((fd_tutorial = ff->addFilter(fm, &filterDef_tutorial, sizeof(FilterDefinition))) == 0)
		return 1;

	vdfd_ver = VIRTUALDUB_FILTERDEF_VERSION;
	vdfd_compat = VIRTUALDUB_FILTERDEF_COMPATIBLE;

	return 0;
}

void VirtualdubFilterModuleDeinit(FilterModule *fm, const FilterFunctions *ff) {
	ff->removeFilter(fd_tutorial);
}

int MyInitProc(FilterActivation *fa, const FilterFunctions *ff) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	mfd->softening = 10;

	return 0;
}

BOOL CALLBACK ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam) {
	MyFilterData *mfd = (MyFilterData *)GetWindowLong(hdlg, DWL_USER);

	switch(msg) {
		case WM_INITDIALOG:
			SetWindowLong(hdlg, DWL_USER, lParam);
			mfd = (MyFilterData *)lParam;

			SetDlgItemInt(hdlg, IDC_SOFTENING, mfd->softening, FALSE);

			return TRUE;

		case WM_COMMAND:
			switch(LOWORD(wParam))
			{
			case IDOK:
				EndDialog(hdlg, 0);
				return TRUE;
			case IDCANCEL:
				EndDialog(hdlg, 1);
				return TRUE;
			case IDC_SOFTENING:
				mfd->softening = GetDlgItemInt(hdlg, IDC_SOFTENING, &mfd->softening, FALSE);
				if (mfd->softening > MAX_SOFTENING)
				{
					mfd->softening = MAX_SOFTENING;
					SetDlgItemInt(hdlg, IDC_SOFTENING, mfd->softening, FALSE);
				}
				break;
			}
			break;
	}

	return FALSE;
}

int ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd)
{
	MyFilterData *mfd = (MyFilterData *) fa->filter_data;
	MyFilterData mfd_old = *mfd;
	int ret;
    extern void Doit(void);

	if (DialogBoxParam(fa->filter->module->hInstModule,
			MAKEINTRESOURCE(IDD_FILTER), hwnd,
			(DLGPROC) ConfigDlgProc, (LPARAM) mfd))
	{
		*mfd = mfd_old;
		ret = TRUE;
	}
	else
	{
		ret = FALSE;
	}
	return(ret);
}

void StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str) {
	MyFilterData *mfd = (MyFilterData *)fa->filter_data;

	sprintf(str, " (softening %d)",
				mfd->softening);
}
